using System;
using System.Data;
using System.Text;
using System.Collections;
using System.Reflection;

namespace gov.va.med.vbecs.BOL
{

		#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>John Vrooland</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>12/11/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// UnitFilter provides a DataView SQL-like object to search and
	/// filter a DataTable of BloodUnits provided by a user. It reduces the
	/// need for dynamic SQL and multiple trips to the database.
	/// </summary>

		#endregion


	public class UnitFilter
	{

        /// <summary>
        ///  Min columns const for VerifyMinColumnsInDataTable
        /// </summary>
        public const string MIN_COLUMNS = "LabelUnitId,UnitProductCode,ProductTypeCode,ProductName,RhFactorCode,BloodTypeCode,AboRh,UnitInDate,UnitExpirationDate,UnitDivisionId";
        //
		private System.Data.DataTable _bloodUnitsInitial;
		private System.Data.DataTable _bloodUnitsSelected;
		private ArrayList _filterAboRhArray= new ArrayList();
		private string _filterExpirationAfterDate = "";
		private string _filterExpirationBeforeDate = "";

		//
		private ArrayList _filterProductTypeArray= new ArrayList();
		private string _filterReceivedAfterDate = "";

		//
		private string _filterReceivedBeforeDate = "";
		private ArrayList _filterUnitGuidArray = new ArrayList();

		//
        bool _sort = true;

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1033"> 
		///		<ExpectedInput>Valid data tables of initial units and selected units</ExpectedInput>
		///		<ExpectedOutput>Valid UnitFilter object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2522"> 
		///		<ExpectedInput>Invalid DataTables</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Create a filter object with InitialUnits and SelectedUnits datatables.
		/// </summary>
		/// <param name="initialUnits">DataTable containing Units to be searched.</param>
		/// <param name="selectedUnits">Units already selected. Schemas must match if provided.
		/// </param>
		public UnitFilter(System.Data.DataTable initialUnits, System.Data.DataTable selectedUnits)
		{
			InitialUnits = initialUnits;
			_bloodUnitsInitial.Columns[Common.VbecsTables.BloodUnit.BloodUnitGuid].Unique = true;
			SelectedUnits = selectedUnits;
			if (_bloodUnitsSelected == null) 
			{
				_bloodUnitsSelected = _bloodUnitsInitial.Clone();
				DataColumn[] dcP =  {_bloodUnitsSelected.Columns[Common.VbecsTables.BloodUnit.BloodUnitGuid]};
				_bloodUnitsSelected.PrimaryKey = dcP;
			}
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1034"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid UnitFilter object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1301"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Create an empty filter object. 
		/// </summary>
		public UnitFilter(bool sort)
		{
			_sort = sort; 

			DataTable bloodUnitsInitial = new DataTable("EmptyInitialTable");
			foreach (string column in MIN_COLUMNS.Split(',') )
			{
				bloodUnitsInitial.Columns.Add(column);
			}
			DataTable bloodUnitSelected = bloodUnitsInitial.Clone();
			_bloodUnitsInitial = bloodUnitsInitial;
			_bloodUnitsSelected = bloodUnitSelected;
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="936"> 
		///		<ExpectedInput>Valid ABO/Rh string</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2523"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Filter on AboRh criteria.
		/// </summary>
		/// <param name="aboRh"></param>
		public void AddAboRhToFilter(string aboRh)
		{
            if (aboRh == null || aboRh.Replace("Pooled","Mx").Length > 7)
            {
                throw new ArgumentException("Invalid Argument:","aboRh");
            }

			aboRh = aboRh.Trim();
			string filterString;
			aboRh = aboRh.Replace("Pooled","Mx");

			if (aboRh.Length > 2)
			{
				filterString = "AboRh = '" + aboRh + "'";
			}
			else
			{
				filterString = "BloodTypeCode = '" + aboRh + "' AND RhFactorCode = 'Z'";
			}
			AddToFilter(_filterAboRhArray, filterString);
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="939"> 
		///		<ExpectedInput>Valid blood unit GUID</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2524"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Filter on CMV negative (Cyto-megalo-virus)?
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void AddCMVToFilter(System.Guid bloodUnitGuid)
		{
			AddToFilter(_filterUnitGuidArray,"BloodUnitGuid ='" + bloodUnitGuid.ToString() + "'");
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="943">
		///		<ExpectedInput>Valid date</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1452"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Filter on units expiring after a given date.
		/// </summary>
		/// <param name="expirationDate"></param>
		public void AddExpirationAfterDateToFilter(DateTime expirationDate)
		{
			_filterExpirationAfterDate = "UnitExpirationDate > #" + expirationDate.ToShortDateString() + "#";
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="944"> 
		///		<ExpectedInput>Valid date</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2525"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Filter on units expiring before a given date.
		/// </summary>
		/// <param name="expirationDate"></param>
		public void AddExpirationBeforeDateToFilter(DateTime expirationDate)
		{
			_filterExpirationBeforeDate = "UnitExpirationDate < #" + expirationDate.ToShortDateString() + "#";
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1031"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2526"> 
		///		<ExpectedInput>Invalid string</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///  Filter on product type criteria.
		/// </summary>
		/// <param name="productTypeCode"></param>
		public void AddProductTypeCodeToFilter(string productTypeCode)
		{
            if (productTypeCode == null || productTypeCode.Length != 4)
            {
                throw new ArgumentException("Invalid Argument:", "productTypeCode");
            }
			AddToFilter(_filterProductTypeArray,"ProductTypeCode ='" + productTypeCode + "'");
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="942"> 
		///		<ExpectedInput>Valid date</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///	<Case type="1" testid ="2508"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Filter on units received after the date parameter
		/// </summary>
		/// <param name="receivedDate"></param>
		public void AddReceivedAfterDateToFilter(DateTime receivedDate)
		{
			_filterReceivedAfterDate = "ShipmentDate > #" + receivedDate.ToShortDateString() + "#";
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="941"> 
		///		<ExpectedInput>Valid date</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2527"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Filter on units received before the date passed.
		/// </summary>
		/// <param name="receivedDate"></param>
		/// 
		public void AddReceivedBeforeDateToFilter(DateTime receivedDate)
		{
			_filterReceivedBeforeDate = "ShipmentDate < #" + receivedDate.ToShortDateString() + "#";
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="937"> 
		///		<ExpectedInput>Valid blood unit GUID</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2528"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Filter on SickleCell negative units, from datatable containing 
		/// all BloodUnitGuids tested for Sickle Cell
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void AddSickleCellToFilter(System.Guid bloodUnitGuid)
		{
			AddToFilter(_filterUnitGuidArray,"BloodUnitGuid ='" + bloodUnitGuid.ToString() + "'");
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="940"> 
		///		<ExpectedInput>Valid blood unit GUID</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2529"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Filter on units with special testing performed.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void AddSpecialTestingToFilter(System.Guid bloodUnitGuid)
		{
			AddToFilter(_filterUnitGuidArray,"BloodUnitGuid ='" + bloodUnitGuid.ToString() + "'");
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		/// <summary>
		/// Add the filter value to the designated filter arraylist.
		/// </summary>
		/// <param name="filterArray">Filter array to add the criterion to.</param>
		/// <param name="filterString">Filter string and value</param>
		private void AddToFilter(ArrayList filterArray, string filterString)
		{
			filterArray.Add(filterString);
		}

#if DEBUG
        //
		private void Dump(string what, ArrayList al)
		{
			System.Diagnostics.Debug.WriteLine("\t\t" + what);
			for (int n = 0 ; n < al.Count ; n++ )
			{
				System.Diagnostics.Debug.WriteLine(al[n]);
			}
		}

		private void DumpAll()
		{
			Dump("_filterProductTypeArray",_filterProductTypeArray);
			Dump("_filterAboRhArray",_filterAboRhArray);
			Dump("_filterUnitGuidArray",_filterUnitGuidArray);
			System.Diagnostics.Debug.WriteLine("_filterReceivedBeforeDate:" + _filterReceivedBeforeDate);
			System.Diagnostics.Debug.WriteLine("_filterReceivedAfterDate:" + _filterReceivedAfterDate);
			System.Diagnostics.Debug.WriteLine("_filterExpirationAfterDate:" + _filterExpirationAfterDate);
			System.Diagnostics.Debug.WriteLine("_filterExpirationBeforeDate:" + _filterExpirationBeforeDate);
		}
        //
#endif

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1028"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataView of filtered units</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1303"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DataView containing the Initial DataTable, apply the filter criteria.
		/// </summary>
		public DataView FilteredUnitsView
		{
			get 
			{
				DataView filteredUnits = new DataView(_bloodUnitsInitial);
				if (_sort)
				{
					filteredUnits.Sort = "ProductName, AboRh, UnitExpirationDate";
				}
				filteredUnits.RowFilter = this.GetFilter();
				return filteredUnits;
			}
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		/// <summary>
		/// Retrieves the filter values from the filter arrays and applies
		/// them to the DataView
		/// </summary>
		/// <returns></returns>
		private string GetFilter()
		{
			string filterString = "";			
			StringBuilder sb = new StringBuilder(); 
			//
			IEnumerator iter;
			if (_filterProductTypeArray.Count > 0 )
			{
				iter = _filterProductTypeArray.GetEnumerator();
				StringBuilder sbProduct = new StringBuilder(); 
				while(iter.MoveNext())
				{
					string filterValue = iter.Current.ToString().Trim();
					if (filterValue != "") 
					{
						if (sbProduct.Length != 0) 
						{
							sbProduct.Append(" OR " + filterValue);  
						}
						else
						{
							sbProduct.Append(filterValue + " ");  
						}
					}
				}
				sb.Append("(" + sbProduct.ToString() + ")");
			}
			if (_filterAboRhArray.Count > 0) 
			{
				if (sb.Length != 0)
				{
					sb.Append(" AND ");
				}
				StringBuilder sbAboRh = new StringBuilder(); 
				iter = _filterAboRhArray.GetEnumerator();
				while(iter.MoveNext())
				{
					string filterValue = iter.Current.ToString().Trim();
					if (filterValue != "") 
					{
						if (sbAboRh.Length != 0) 
						{
							sbAboRh.Append(" OR " + filterValue);  
						}
						else
						{
							sbAboRh.Append(filterValue + " ");  
						}
					}
				}
				sb.Append("(" + sbAboRh.ToString() + ")");
			}
			if (_filterUnitGuidArray.Count > 0) 
			{
				if (sb.Length != 0)
				{
					sb.Append(" AND ");
				}
				StringBuilder sbGuid = new StringBuilder(); 
				iter = _filterUnitGuidArray.GetEnumerator();
				while(iter.MoveNext())
				{
					string filterValue = iter.Current.ToString().Trim();
					if (filterValue != "") 
					{
						if (sbGuid.Length != 0) 
						{
							sbGuid.Append(" OR " + filterValue);  
						}
						else
						{
							sbGuid.Append(filterValue + " ");  
						}
					}
				}
				sb.Append("(" + sbGuid.ToString() + ")");
			}
			if (_filterReceivedAfterDate != "") 
			{
				if (sb.Length != 0)
				{
					sb.Append(" AND ");
				}
				sb.Append(_filterReceivedAfterDate);
			}
			if (_filterReceivedBeforeDate != "") 
			{
				if (sb.Length != 0)
				{
					sb.Append(" AND ");
				}
				sb.Append(_filterReceivedBeforeDate);
			}
			if (_filterExpirationAfterDate != "") 
			{
				if (sb.Length != 0)
				{
					sb.Append(" AND ");
				}
				sb.Append("(" + _filterExpirationAfterDate + ")");
			}
			if (_filterExpirationBeforeDate != "") 
			{
				if (sb.Length != 0)
				{
					sb.Append(" AND ");
				}
				sb.Append("(" + _filterExpirationBeforeDate + ")");
			}
			filterString = sb.ToString();
			return filterString;
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1032"> 
		///		<ExpectedInput>Valid DataTable</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1341"> 
		///		<ExpectedInput>Invalid DataTable</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sets and gets the InitialUnits DataTable used to filter.
		/// </summary>
		public DataTable InitialUnits
		{
			get 
			{
				return _bloodUnitsInitial;
			}
			set 
			{
				string[] columnList = MIN_COLUMNS.Split(',');
				if (VerifyMinColumnsInDataTable(columnList, value) )
				{
					_bloodUnitsInitial = value;
				}
				else
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
											Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
											value).ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="946"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2530"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Clear the Abo/Rh filter.
		/// </summary>
		/// <param name="aboRh"></param>
		public void RemoveAboRhFromFilter(string aboRh)
		{
			aboRh = aboRh.Trim();
			aboRh = aboRh.Replace("Pooled","Mx");

			string filterString;
			aboRh = aboRh.Replace("Pooled","Mx");

			if (aboRh.Length > 3)
			{
				filterString = "AboRh = '" + aboRh + "'";
			}
			else
			{
				filterString = "BloodTypeCode = '" + aboRh + "' AND RhFactorCode = 'Z'";
			}

			RemoveFromFilter(_filterAboRhArray,filterString);
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="948"> 
		///		<ExpectedInput>Valid Blood Unit GUID</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2531"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Clear the CMV filter.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void RemoveCMVFromFilter(System.Guid bloodUnitGuid)
		{
			RemoveFromFilter(_filterUnitGuidArray,"BloodUnitGuid ='" + bloodUnitGuid + "'");
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="952"> 
		///		<ExpectedInput>Valid Date</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2532"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Clear the expires after date criterion.
		/// </summary>
		/// <param name="expirationDate"></param>
		public void RemoveExpirationAfterDateFromFilter(DateTime expirationDate)
		{
			_filterExpirationAfterDate = "";
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="953"> 
		///		<ExpectedInput>Valid Date</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2533"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Clear the expires before criterion.
		/// </summary>
		/// <param name="expirationDate"></param>
		public void RemoveExpirationBeforeDateFromFilter(DateTime expirationDate)
		{
			_filterExpirationBeforeDate = "";
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		/// <summary>
		/// Remove the filter value from the array specified.
		/// </summary>
		/// <param name="filterArray"></param>
		/// <param name="filterString"></param>
		private void RemoveFromFilter(ArrayList filterArray, string filterString)
		{
			filterArray.Remove(filterString);
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="945"> 
		///		<ExpectedInput>Valid product code string</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2534"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Clear the product type filter.
		/// </summary>
		/// <param name="productTypeCode"></param>
		public void RemoveProductTypeCodeFromFilter(string productTypeCode)
		{
			RemoveFromFilter(_filterProductTypeArray,"ProductTypeCode ='" + productTypeCode + "'");
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="951"> 
		///		<ExpectedInput>Valid Date</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2535"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Clear the received after date filter.
		/// </summary>
		/// <param name="receivedDate"></param>
		public void RemoveReceivedAfterDateFromFilter(DateTime receivedDate)
		{
			_filterReceivedAfterDate = "";
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="950"> 
		///		<ExpectedInput>Valid Date</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2536"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Clear the received before date filter criterion.
		/// </summary>
		/// <param name="receivedDate"></param>
		public void RemoveReceivedBeforeDateFromFilter(DateTime receivedDate)
		{
			_filterReceivedBeforeDate = "";
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="947"> 
		///		<ExpectedInput>Valid Blood Unit GUID</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2537"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Clear the sickle cell filter.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void RemoveSickleCellFromFilter(System.Guid bloodUnitGuid)
		{
			RemoveFromFilter(_filterUnitGuidArray,"BloodUnitGuid ='" + bloodUnitGuid + "'");
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="949"> 
		///		<ExpectedInput>Valid Blood Unit GUID</ExpectedInput>
		///		<ExpectedOutput>No exceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2538"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Clear the special testing filter.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void RemoveSpecialTestingFromFilter(System.Guid bloodUnitGuid)
		{
			RemoveFromFilter(_filterUnitGuidArray,"BloodUnitGuid ='" + bloodUnitGuid + "'");
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1029"> 
		///		<ExpectedInput>Valid DataTable</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1342"> 
		///		<ExpectedInput>Inalid DataTable</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DataTable providing storage of units selected from the Initial DataTable.
		/// </summary>
		public System.Data.DataTable SelectedUnits
		{
			get
			{
				return _bloodUnitsSelected;
			}
			set 
			{
				if (value != null)
				{
					if (!VerifySelectedMatchesInitial(value)) 
					{
						throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.TableSchemaMismatch().ResString);
					}
				}
				_bloodUnitsSelected = value;
			}
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1027"> 
		///		<ExpectedInput>string array of column names, Valid data table</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2539"> 
		///		<ExpectedInput>string array of column names, Invalid data table</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Verify specific set of columns are in the DataTable provided. 
		/// </summary>
		/// <param name="colNames"></param>
		/// <param name="dtTable"></param>
		/// <returns></returns>
		public bool VerifyMinColumnsInDataTable(string [] colNames, System.Data.DataTable dtTable)
		{
			return Common.Utility.VerifyMinColumnsInDataTable(colNames, dtTable);
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/11/2002</CreationDate>
		/// <summary>
		/// Verify the schema of the selected units DataTable matches the initial units DataTable.
		/// </summary>
		/// <param name="dt">Selected Units table to compare</param>
		/// <returns>bool</returns>
		private bool VerifySelectedMatchesInitial(DataTable dt)
		{
			if (InitialUnits == null  )
			{
				throw new BusinessObjectException("Invalid Initial table value: null");
			}
			foreach (DataColumn dc in dt.Columns )
			{
				if (!InitialUnits.Columns.Contains(dc.ColumnName))
				{
					return false;
				}
			}
			return true;
		}
		//
	}

}
